<?php
/**
 * Plugin Name:       Cargador de Páginas Personalizado HDVF
 * Plugin URI:        https://tudominio.com
 * Description:       Añade un cargador animado entre páginas, estilos personalizados para el login y el admin, configurable desde el panel de administración.
 * Version:           1.0.0
 * Author:            HDVF
 * Author URI:        https://tudominio.com
 * License:           GPL-2.0+
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       mi-plugin-cargador
 * Domain Path:       /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Clase principal del Plugin Cargador de Páginas Personalizado
 */
class HDVF_Page_Loader_Plugin {

    public function __construct() {
        // Añadir las acciones y filtros para el front-end y back-end
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_styles_scripts' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) ); // Estilos para el color picker en admin
        add_action( 'wp_footer', array( $this, 'render_page_loader' ) );

        // Estilos y personalizaciones del login
        add_action( 'login_enqueue_scripts', array( $this, 'custom_login_styles' ) );
        add_filter( 'login_headerurl', fn() => home_url() );
        add_filter( 'login_headertext', fn() => get_bloginfo('name') );
        add_action( 'login_footer', array( $this, 'custom_login_footer' ) );

        // Estilos y personalizaciones del admin
        add_filter( 'admin_footer_text', array( $this, 'custom_admin_footer' ) );
        add_action( 'admin_head', array( $this, 'custom_admin_styles' ) );

        // Opciones del panel de administración
        add_action( 'admin_menu', array( $this, 'add_plugin_admin_page' ) );
        add_action( 'admin_init', array( $this, 'register_plugin_settings' ) );
    }

    /**
     * Encola los estilos y scripts necesarios para el cargador en el front-end.
     */
    public function enqueue_styles_scripts() {
        // No necesitamos encolar archivos si todo el CSS/JS es inline en wp_footer.
        // Pero si en el futuro tienes archivos, los encolarías aquí.
    }

    /**
     * Encola los estilos necesarios para el color picker en el admin.
     */
    public function enqueue_admin_styles() {
        if ( isset( $_GET['page'] ) && $_GET['page'] == 'hdvf-page-loader-settings' ) {
            wp_enqueue_style( 'wp-color-picker' );
            wp_enqueue_script( 'wp-color-picker' );
        }
    }

    /**
     * Renderiza el HTML y JS del cargador de páginas en el footer.
     */
    public function render_page_loader() {
        $options = get_option( 'hdvf_page_loader_options' );
        $loader_active = isset( $options['loader_active'] ) ? $options['loader_active'] : 'off';

        if ( 'off' === $loader_active ) {
            return; // No mostrar el loader si está desactivado.
        }

        $loader_color = isset( $options['loader_color'] ) ? sanitize_hex_color( $options['loader_color'] ) : '#C1262E';
        $loader_type = isset( $options['loader_type'] ) ? sanitize_text_field( $options['loader_type'] ) : 'spinner';
        $background_opacity = isset( $options['background_opacity'] ) ? floatval( $options['background_opacity'] ) : 0.85;
        $background_color = isset( $options['background_color'] ) ? sanitize_hex_color( $options['background_color'] ) : '#ffffff';

        // Determinar el contenido del spinner según el tipo seleccionado
        $spinner_html = '';
        switch ( $loader_type ) {
            case 'spinner':
                $spinner_html = '<div class="custom-spinner"></div>';
                break;
            case 'dots':
                $spinner_html = '
                    <div class="custom-dots">
                        <div class="dot"></div>
                        <div class="dot"></div>
                        <div class="dot"></div>
                    </div>
                ';
                break;
            case 'bars':
                $spinner_html = '
                    <div class="custom-bars">
                        <div class="bar"></div>
                        <div class="bar"></div>
                        <div class="bar"></div>
                        <div class="bar"></div>
                        <div class="bar"></div>
                    </div>
                ';
                break;
            // Puedes añadir más tipos aquí
            default:
                $spinner_html = '<div class="custom-spinner"></div>';
        }

        ?>
        <style>
            /* Estilos generales del loader */
            #custom-page-loader {
                position: fixed;
                top: 0;
                left: 0;
                width: 100vw;
                height: 100vh;
                background-color: <?php echo $background_color; ?>;
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 9999;
                opacity: 0;
                pointer-events: none;
                transition: opacity 0.3s ease;
            }

            #custom-page-loader.show {
                opacity: <?php echo $background_opacity; ?>;
                pointer-events: auto;
            }

            /* Estilos para el Spinner (tipo por defecto y 'spinner') */
            .custom-spinner {
                width: 50px;
                height: 50px;
                border: 6px solid <?php echo $loader_color; ?>;
                border-top: 6px solid transparent;
                border-radius: 50%;
                animation: spin 1s linear infinite;
            }

            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }

            /* Estilos para el tipo 'dots' */
            .custom-dots {
                display: flex;
                justify-content: space-around;
                width: 80px;
            }
            .custom-dots .dot {
                width: 15px;
                height: 15px;
                background-color: <?php echo $loader_color; ?>;
                border-radius: 50%;
                animation: bounce 1.4s infinite ease-in-out both;
            }
            .custom-dots .dot:nth-child(1) { animation-delay: -0.32s; }
            .custom-dots .dot:nth-child(2) { animation-delay: -0.16s; }
            @keyframes bounce {
                0%, 80%, 100% { transform: scale(0); }
                40% { transform: scale(1.0); }
            }

            /* Estilos para el tipo 'bars' */
            .custom-bars {
                display: flex;
                height: 50px;
                align-items: flex-end;
            }
            .custom-bars .bar {
                width: 10px;
                height: 5px;
                margin: 0 2px;
                background-color: <?php echo $loader_color; ?>;
                animation: stretch 1.2s infinite ease-in-out;
            }
            .custom-bars .bar:nth-child(1) { animation-delay: -1.2s; }
            .custom-bars .bar:nth-child(2) { animation-delay: -1.1s; }
            .custom-bars .bar:nth-child(3) { animation-delay: -1.0s; }
            .custom-bars .bar:nth-child(4) { animation-delay: -0.9s; }
            .custom-bars .bar:nth-child(5) { animation-delay: -0.8s; }
            @keyframes stretch {
                0%, 40%, 100% { transform: scaleY(0.4); }
                20% { transform: scaleY(1.0); }
            }
        </style>

        <div id="custom-page-loader">
            <?php echo $spinner_html; // Imprimir el HTML del spinner seleccionado ?>
        </div>

        <script>
            document.addEventListener('DOMContentLoaded', function () {
                const loader = document.getElementById('custom-page-loader');

                document.querySelectorAll('a[href]').forEach(link => {
                    const href = link.getAttribute('href');
                    const currentDomain = window.location.hostname;
                    const linkDomain = (new URL(link.href)).hostname;

                    // Mostrar loader al hacer clic en enlaces internos del mismo dominio
                    if (
                        href &&
                        !href.startsWith('#') && // No es un ancla
                        !href.startsWith('mailto:') && // No es un email
                        !href.startsWith('tel:') && // No es un teléfono
                        link.target !== '_blank' && // No abre en nueva pestaña
                        !event.metaKey && !event.ctrlKey && !event.shiftKey && !event.altKey && // No se abren en nueva ventana/pestaña con teclas modificadoras
                        (linkDomain === currentDomain || !link.hostname) // Es un enlace interno o relativo
                    ) {
                        link.addEventListener('click', function (e) {
                            // Evitar el comportamiento por defecto para dar tiempo a que se muestre el loader
                            e.preventDefault();
                            loader.classList.add('show');
                            // Redirigir después de un pequeño retraso
                            setTimeout(() => {
                                window.location.href = href;
                            }, 100); // Pequeño retardo para que la transición de opacidad sea visible
                        });
                    }
                });

                // Ocultar loader al cargar la página
                window.addEventListener('load', () => {
                    loader.classList.remove('show');
                });
            });
        </script>
        <?php
    }

    /**
     * Estilos visuales personalizados del login.
     */
    public function custom_login_styles() {
        $options = get_option( 'hdvf_page_loader_options' );
        $accent_color = isset( $options['accent_color'] ) ? sanitize_hex_color( $options['accent_color'] ) : '#D63D23';
        $login_bg_color = isset( $options['login_bg_color'] ) ? sanitize_hex_color( $options['login_bg_color'] ) : '#111';
        $login_form_bg_color = isset( $options['login_form_bg_color'] ) ? sanitize_hex_color( $options['login_form_bg_color'] ) : '#1b1b1b';
        $login_text_color = isset( $options['login_text_color'] ) ? sanitize_hex_color( $options['login_text_color'] ) : '#fff';
        $login_label_color = isset( $options['login_label_color'] ) ? sanitize_hex_color( $options['login_label_color'] ) : '#black'; // Ojo con 'black', mejor un hex
        $login_input_bg_color = isset( $options['login_input_bg_color'] ) ? sanitize_hex_color( $options['login_input_bg_color'] ) : '#292929';
        $login_input_border_color = isset( $options['login_input_border_color'] ) ? sanitize_hex_color( $options['login_input_border_color'] ) : '#444';
        $login_link_color = isset( $options['login_link_color'] ) ? sanitize_hex_color( $options['login_link_color'] ) : '#f5f5f5';

        ?>
        <style>
        body.login {
            background-color: <?php echo $login_bg_color; ?> !important;
            color: <?php echo $login_text_color; ?> !important;
        }

        body.login #login {
            background-color: <?php echo $login_form_bg_color; ?> !important;
            padding: 30px 25px;
            border-radius: 12px;
            box-shadow: 0 0 25px rgba(255,255,255,0.05);
        }

        body.login h1 a {
            background-image: url('<?php echo get_site_icon_url(); ?>');
            background-size: contain;
            width: 100%;
            height: 80px;
        }

        body.login label {
            color: <?php echo $login_label_color; ?> !important;
        }

        body.login input[type="text"],
        body.login input[type="password"],
        body.login input[type="email"] {
            background: <?php echo $login_input_bg_color; ?>;
            border: 1px solid <?php echo $login_input_border_color; ?>;
            color: <?php echo $login_text_color; ?>;
            border-radius: 6px;
        }

        body.login input:focus {
            border-color: <?php echo $accent_color; ?>;
            box-shadow: 0 0 6px <?php echo $accent_color; ?>;
        }

        .login .button-primary {
            background-color: <?php echo $accent_color; ?> !important;
            border: none;
            box-shadow: 0 0 10px rgba(<?php echo hex_to_rgb( $accent_color ); ?>, 0.6);
            color: white;
            border-radius: 6px;
        }

        .login .button-primary:hover {
            background-color: <?php echo adjust_brightness( $accent_color, -20 ); ?> !important; /* Oscurecer un poco al pasar el ratón */
        }

        .login #backtoblog a,
        .login #nav a {
            color: <?php echo $login_link_color; ?> !important;
        }

        .login #login_error,
        .login .message,
        .login .success {
            background-color: <?php echo adjust_brightness( $login_form_bg_color, -10 ); ?>; /* Un poco más oscuro que el fondo del formulario */
            border-left: 4px solid <?php echo $accent_color; ?>;
            color: <?php echo $login_text_color; ?>;
        }

        .login .privacy-policy-page-link {
            color: <?php echo adjust_brightness( $login_link_color, -20 ); ?> !important;
        }

        .login-footer-hdvf {
            margin-top: 30px;
            font-size: 13px;
            color: <?php echo adjust_brightness( $login_text_color, -30 ); ?>;
            text-align: center;
        }
        </style>
        <?php
    }

    /**
     * Footer personalizado login.
     */
    public function custom_login_footer() {
        echo '<div class="login-footer-hdvf">Sitio desarrollado por <strong>HDF</strong><br><em>CMS WordPress Custom by HDVF</em></div>';
    }

    /**
     * Footer admin.
     */
    public function custom_admin_footer() {
        return '✨ <strong>CMS WordPress Custom by HDVF</strong> | Sitio desarrollado por HDF';
    }

    /**
     * Admin panel estilo (coherente con login).
     */
    public function custom_admin_styles() {
        $options = get_option( 'hdvf_page_loader_options' );
        $accent_color = isset( $options['accent_color'] ) ? sanitize_hex_color( $options['accent_color'] ) : '#D63D23';
        $admin_bar_bg_color = isset( $options['admin_bar_bg_color'] ) ? sanitize_hex_color( $options['admin_bar_bg_color'] ) : '#111';
        $admin_menu_bg_color = isset( $options['admin_menu_bg_color'] ) ? sanitize_hex_color( $options['admin_menu_bg_color'] ) : '#1c1c1c';
        $admin_menu_link_color = isset( $options['admin_menu_link_color'] ) ? sanitize_hex_color( $options['admin_menu_link_color'] ) : '#ccc';
        $admin_heading_color = isset( $options['admin_heading_color'] ) ? sanitize_hex_color( $options['admin_heading_color'] ) : '#222';
        $admin_footer_color = isset( $options['admin_footer_color'] ) ? sanitize_hex_color( $options['admin_footer_color'] ) : '#555';

        echo '<style>
            #wpadminbar {
                background-color: ' . $admin_bar_bg_color . ' !important;
            }

            #adminmenu, #adminmenu .wp-submenu, #adminmenu .wp-has-current-submenu .wp-submenu {
                background-color: ' . $admin_menu_bg_color . ' !important;
            }

            #adminmenu a {
                color: ' . $admin_menu_link_color . ' !important;
            }

            #adminmenu a:hover, #adminmenu li.menu-top:hover > a {
                background-color: ' . $accent_color . ' !important;
                color: #fff !important;
            }

            .wrap h1 {
                border-left: 5px solid ' . $accent_color . ';
                padding-left: 10px;
                color: ' . $admin_heading_color . ';
            }

            #wpfooter {
                text-align: center;
                font-size: 13px;
                color: ' . $admin_footer_color . ';
            }
        </style>';
    }

    /**
     * Añade la página de opciones del plugin al menú de administración.
     */
    public function add_plugin_admin_page() {
        add_options_page(
            'Opciones Cargador HDVF', // Título de la página
            'Cargador HDVF',          // Texto del menú
            'manage_options',         // Capacidad requerida
            'hdvf-page-loader-settings', // Slug del menú
            array( $this, 'render_plugin_settings_page' ) // Callback para renderizar la página
        );
    }

    /**
     * Registra las configuraciones del plugin.
     */
    public function register_plugin_settings() {
        register_setting(
            'hdvf_page_loader_options_group', // Nombre del grupo de opciones
            'hdvf_page_loader_options',       // Nombre de la opción en la base de datos
            array( $this, 'sanitize_plugin_options' ) // Callback para sanear los datos
        );

        // Sección para el Loader
        add_settings_section(
            'hdvf_loader_settings_section',    // ID de la sección
            'Configuración del Cargador de Páginas', // Título de la sección
            array( $this, 'loader_section_callback' ), // Callback de la sección
            'hdvf-page-loader-settings'        // Página a la que pertenece
        );

        add_settings_field(
            'loader_active',                   // ID del campo
            'Activar Cargador',                // Etiqueta del campo
            array( $this, 'loader_active_callback' ), // Callback del campo
            'hdvf-page-loader-settings',       // Página a la que pertenece
            'hdvf_loader_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'loader_type',                     // ID del campo
            'Tipo de Cargador',                // Etiqueta del campo
            array( $this, 'loader_type_callback' ), // Callback del campo
            'hdvf-page-loader-settings',       // Página a la que pertenece
            'hdvf_loader_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'loader_color',                    // ID del campo
            'Color Principal del Cargador',    // Etiqueta del campo
            array( $this, 'loader_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',       // Página a la que pertenece
            'hdvf_loader_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'background_color',                // ID del campo
            'Color de Fondo del Cargador',     // Etiqueta del campo
            array( $this, 'background_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',       // Página a la que pertenece
            'hdvf_loader_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'background_opacity',              // ID del campo
            'Opacidad del Fondo',              // Etiqueta del campo
            array( $this, 'background_opacity_callback' ), // Callback del campo
            'hdvf-page-loader-settings',       // Página a la que pertenece
            'hdvf_loader_settings_section'     // Sección a la que pertenece
        );

        // Sección para los estilos del Login
        add_settings_section(
            'hdvf_login_settings_section',    // ID de la sección
            'Configuración de Estilos del Login', // Título de la sección
            array( $this, 'login_section_callback' ), // Callback de la sección
            'hdvf-page-loader-settings'        // Página a la que pertenece
        );

        add_settings_field(
            'accent_color',                   // ID del campo
            'Color Principal/Acento',         // Etiqueta del campo
            array( $this, 'accent_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_login_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'login_bg_color',                 // ID del campo
            'Color de Fondo del Login',       // Etiqueta del campo
            array( $this, 'login_bg_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_login_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'login_form_bg_color',            // ID del campo
            'Color de Fondo del Formulario',  // Etiqueta del campo
            array( $this, 'login_form_bg_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_login_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'login_text_color',               // ID del campo
            'Color de Texto Principal',       // Etiqueta del campo
            array( $this, 'login_text_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_login_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'login_label_color',              // ID del campo
            'Color de las Etiquetas',         // Etiqueta del campo
            array( $this, 'login_label_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_login_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'login_input_bg_color',           // ID del campo
            'Color de Fondo de los Inputs',   // Etiqueta del campo
            array( $this, 'login_input_bg_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_login_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'login_input_border_color',       // ID del campo
            'Color de Borde de los Inputs',   // Etiqueta del campo
            array( $this, 'login_input_border_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_login_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'login_link_color',               // ID del campo
            'Color de los Enlaces del Login', // Etiqueta del campo
            array( $this, 'login_link_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_login_settings_section'     // Sección a la que pertenece
        );


        // Sección para los estilos del Admin
        add_settings_section(
            'hdvf_admin_settings_section',    // ID de la sección
            'Configuración de Estilos del Panel de Administración', // Título de la sección
            array( $this, 'admin_section_callback' ), // Callback de la sección
            'hdvf-page-loader-settings'        // Página a la que pertenece
        );

        add_settings_field(
            'admin_bar_bg_color',             // ID del campo
            'Color de Fondo de la Barra Admin', // Etiqueta del campo
            array( $this, 'admin_bar_bg_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_admin_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'admin_menu_bg_color',            // ID del campo
            'Color de Fondo del Menú Admin',  // Etiqueta del campo
            array( $this, 'admin_menu_bg_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_admin_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'admin_menu_link_color',          // ID del campo
            'Color de los Enlaces del Menú Admin', // Etiqueta del campo
            array( $this, 'admin_menu_link_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_admin_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'admin_heading_color',            // ID del campo
            'Color de Títulos del Admin',     // Etiqueta del campo
            array( $this, 'admin_heading_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_admin_settings_section'     // Sección a la que pertenece
        );

        add_settings_field(
            'admin_footer_color',             // ID del campo
            'Color del Texto del Footer Admin', // Etiqueta del campo
            array( $this, 'admin_footer_color_callback' ), // Callback del campo
            'hdvf-page-loader-settings',      // Página a la que pertenece
            'hdvf_admin_settings_section'     // Sección a la que pertenece
        );
    }

    /**
     * Callback para la sección del loader.
     */
    public function loader_section_callback() {
        echo '<p>Configura las opciones para el cargador de páginas.</p>';
    }

    /**
     * Callback para la sección de estilos del login.
     */
    public function login_section_callback() {
        echo '<p>Personaliza los colores y estilos de la página de inicio de sesión de WordPress.</p>';
    }

    /**
     * Callback para la sección de estilos del admin.
     */
    public function admin_section_callback() {
        echo '<p>Personaliza los colores y estilos del panel de administración de WordPress.</p>';
    }

    /**
     * Callback para el campo 'loader_active'.
     */
    public function loader_active_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $checked = isset( $options['loader_active'] ) ? checked( $options['loader_active'], 'on', false ) : '';
        echo '<input type="checkbox" id="loader_active" name="hdvf_page_loader_options[loader_active]" value="on" ' . $checked . ' /> Activar cargador entre páginas';
    }

    /**
     * Callback para el campo 'loader_type'.
     */
    public function loader_type_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $selected_type = isset( $options['loader_type'] ) ? $options['loader_type'] : 'spinner';
        ?>
        <select id="loader_type" name="hdvf_page_loader_options[loader_type]">
            <option value="spinner" <?php selected( $selected_type, 'spinner' ); ?>>Spinner (Círculo Giratorio)</option>
            <option value="dots" <?php selected( $selected_type, 'dots' ); ?>>Puntos Rebotando</option>
            <option value="bars" <?php selected( $selected_type, 'bars' ); ?>>Barras Estirándose</option>
            </select>
        <?php
    }

    /**
     * Callback para el campo 'loader_color'.
     */
    public function loader_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['loader_color'] ) ? $options['loader_color'] : '#C1262E';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[loader_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'background_color'.
     */
    public function background_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['background_color'] ) ? $options['background_color'] : '#ffffff';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[background_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'background_opacity'.
     */
    public function background_opacity_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $opacity = isset( $options['background_opacity'] ) ? $options['background_opacity'] : 0.85;
        echo '<input type="number" step="0.05" min="0" max="1" id="background_opacity" name="hdvf_page_loader_options[background_opacity]" value="' . esc_attr( $opacity ) . '" />';
        echo '<p class="description">Valores entre 0 (transparente) y 1 (opaco).</p>';
    }

    /**
     * Callback para el campo 'accent_color'.
     */
    public function accent_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['accent_color'] ) ? $options['accent_color'] : '#D63D23';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[accent_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'login_bg_color'.
     */
    public function login_bg_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['login_bg_color'] ) ? $options['login_bg_color'] : '#111';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[login_bg_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'login_form_bg_color'.
     */
    public function login_form_bg_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['login_form_bg_color'] ) ? $options['login_form_bg_color'] : '#1b1b1b';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[login_form_bg_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'login_text_color'.
     */
    public function login_text_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['login_text_color'] ) ? $options['login_text_color'] : '#fff';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[login_text_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'login_label_color'.
     */
    public function login_label_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['login_label_color'] ) ? $options['login_label_color'] : '#000000';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[login_label_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'login_input_bg_color'.
     */
    public function login_input_bg_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['login_input_bg_color'] ) ? $options['login_input_bg_color'] : '#292929';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[login_input_bg_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'login_input_border_color'.
     */
    public function login_input_border_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['login_input_border_color'] ) ? $options['login_input_border_color'] : '#444';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[login_input_border_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'login_link_color'.
     */
    public function login_link_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['login_link_color'] ) ? $options['login_link_color'] : '#f5f5f5';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[login_link_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'admin_bar_bg_color'.
     */
    public function admin_bar_bg_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['admin_bar_bg_color'] ) ? $options['admin_bar_bg_color'] : '#111';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[admin_bar_bg_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'admin_menu_bg_color'.
     */
    public function admin_menu_bg_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['admin_menu_bg_color'] ) ? $options['admin_menu_bg_color'] : '#1c1c1c';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[admin_menu_bg_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'admin_menu_link_color'.
     */
    public function admin_menu_link_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['admin_menu_link_color'] ) ? $options['admin_menu_link_color'] : '#ccc';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[admin_menu_link_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'admin_heading_color'.
     */
    public function admin_heading_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['admin_heading_color'] ) ? $options['admin_heading_color'] : '#222';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[admin_heading_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Callback para el campo 'admin_footer_color'.
     */
    public function admin_footer_color_callback() {
        $options = get_option( 'hdvf_page_loader_options' );
        $color = isset( $options['admin_footer_color'] ) ? $options['admin_footer_color'] : '#555';
        echo '<input type="text" class="hdvf-color-picker" name="hdvf_page_loader_options[admin_footer_color]" value="' . esc_attr( $color ) . '" />';
    }

    /**
     * Renderiza la página de opciones del plugin.
     */
    public function render_plugin_settings_page() {
        ?>
        <div class="wrap">
            <h1>Configuración del Cargador y Estilos HDVF</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'hdvf_page_loader_options_group' ); // Muestra los campos de seguridad ocultos
                do_settings_sections( 'hdvf-page-loader-settings' ); // Muestra las secciones y campos
                submit_button(); // Muestra el botón de guardar
                ?>
            </form>
        </div>
        <script type="text/javascript">
            jQuery(document).ready(function($){
                $('.hdvf-color-picker').wpColorPicker();
            });
        </script>
        <?php
    }

    /**
     * Sanea los datos del formulario antes de guardarlos.
     *
     * @param array $input Los datos enviados.
     * @return array Los datos saneados.
     */
    public function sanitize_plugin_options( $input ) {
        $sanitized_input = array();

        // Loader settings
        $sanitized_input['loader_active'] = isset( $input['loader_active'] ) ? 'on' : 'off';
        $sanitized_input['loader_type'] = sanitize_text_field( $input['loader_type'] );
        $sanitized_input['loader_color'] = sanitize_hex_color( $input['loader_color'] );
        $sanitized_input['background_color'] = sanitize_hex_color( $input['background_color'] );
        $sanitized_input['background_opacity'] = floatval( $input['background_opacity'] );
        if ( $sanitized_input['background_opacity'] < 0 || $sanitized_input['background_opacity'] > 1 ) {
            $sanitized_input['background_opacity'] = 0.85; // Valor por defecto si es inválido
        }


        // Login settings
        $sanitized_input['accent_color'] = sanitize_hex_color( $input['accent_color'] );
        $sanitized_input['login_bg_color'] = sanitize_hex_color( $input['login_bg_color'] );
        $sanitized_input['login_form_bg_color'] = sanitize_hex_color( $input['login_form_bg_color'] );
        $sanitized_input['login_text_color'] = sanitize_hex_color( $input['login_text_color'] );
        $sanitized_input['login_label_color'] = sanitize_hex_color( $input['login_label_color'] );
        $sanitized_input['login_input_bg_color'] = sanitize_hex_color( $input['login_input_bg_color'] );
        $sanitized_input['login_input_border_color'] = sanitize_hex_color( $input['login_input_border_color'] );
        $sanitized_input['login_link_color'] = sanitize_hex_color( $input['login_link_color'] );

        // Admin settings
        $sanitized_input['admin_bar_bg_color'] = sanitize_hex_color( $input['admin_bar_bg_color'] );
        $sanitized_input['admin_menu_bg_color'] = sanitize_hex_color( $input['admin_menu_bg_color'] );
        $sanitized_input['admin_menu_link_color'] = sanitize_hex_color( $input['admin_menu_link_color'] );
        $sanitized_input['admin_heading_color'] = sanitize_hex_color( $input['admin_heading_color'] );
        $sanitized_input['admin_footer_color'] = sanitize_hex_color( $input['admin_footer_color'] );


        return $sanitized_input;
    }

} // Fin de la clase HDVF_Page_Loader_Plugin

// Instanciar la clase del plugin
new HDVF_Page_Loader_Plugin();


/**
 * Funciones de ayuda para colores.
 */
if ( ! function_exists( 'hex_to_rgb' ) ) {
    /**
     * Convierte un color hexadecimal a valores RGB.
     *
     * @param string $hex Color hexadecimal (ej. #RRGGBB o RRGGBB).
     * @return string Una cadena con los valores RGB separados por comas.
     */
    function hex_to_rgb( $hex ) {
        $hex = str_replace('#', '', $hex);

        if ( strlen( $hex ) == 3 ) {
            $r = hexdec( substr( $hex, 0, 1 ) . substr( $hex, 0, 1 ) );
            $g = hexdec( substr( $hex, 1, 1 ) . substr( $hex, 1, 1 ) );
            $b = hexdec( substr( $hex, 2, 1 ) . substr( $hex, 2, 1 ) );
        } else {
            $r = hexdec( substr( $hex, 0, 2 ) );
            $g = hexdec( substr( $hex, 2, 2 ) );
            $b = hexdec( substr( $hex, 4, 2 ) );
        }
        return "$r, $g, $b";
    }
}

if ( ! function_exists( 'adjust_brightness' ) ) {
    /**
     * Ajusta el brillo de un color hexadecimal.
     *
     * @param string $hex Color hexadecimal (ej. #RRGGBB).
     * @param int    $steps Cantidad de pasos para ajustar (+ para más claro, - para más oscuro).
     * @return string El color hexadecimal ajustado.
     */
    function adjust_brightness( $hex, $steps ) {
        $hex = str_replace('#', '', $hex);

        // Obtenemos los valores RGB
        if ( strlen( $hex ) == 3 ) {
            $r = hexdec( substr( $hex, 0, 1 ) . substr( $hex, 0, 1 ) );
            $g = hexdec( substr( $hex, 1, 1 ) . substr( $hex, 1, 1 ) );
            $b = hexdec( substr( $hex, 2, 1 ) . substr( $hex, 2, 1 ) );
        } else {
            $r = hexdec( substr( $hex, 0, 2 ) );
            $g = hexdec( substr( $hex, 2, 2 ) );
            $b = hexdec( substr( $hex, 4, 2 ) );
        }

        // Ajustar el brillo
        $r = max( 0, min( 255, $r + $steps ) );
        $g = max( 0, min( 255, $g + $steps ) );
        $b = max( 0, min( 255, $b + $steps ) );

        // Convertir de nuevo a hexadecimal
        return '#' . str_pad( dechex( $r ), 2, '0', STR_PAD_LEFT )
                   . str_pad( dechex( $g ), 2, '0', STR_PAD_LEFT )
                   . str_pad( dechex( $b ), 2, '0', STR_PAD_LEFT );
    }
}